# -*- coding: utf-8 -*-
import os
import sqlite3
from flask import Flask, render_template, request, g, redirect, url_for
from datetime import datetime

DATABASE = os.path.join(os.path.dirname(__file__), "data", "products.sqlite")

def get_db():
    db = getattr(g, "_database", None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE)
        db.row_factory = sqlite3.Row
    return db

def query_db(query, args=(), one=False):
    cur = get_db().execute(query, args)
    rv = cur.fetchall()
    cur.close()
    return (rv[0] if rv else None) if one else rv

def create_app():
    app = Flask(__name__)

    @app.teardown_appcontext
    def close_connection(exception):
        db = getattr(g, "_database", None)
        if db is not None:
            db.close()

    @app.route("/", methods=["GET"])
    def index():
        return render_template("index.html")

    @app.route("/search", methods=["GET"])
    def search():
        q = request.args.get("q", "").strip()
        sort = request.args.get("sort", "price")
        order = "ASC" if request.args.get("dir", "asc") == "asc" else "DESC"
        params = []
        where = "1=1"
        if q:
            where += " AND lower(name) LIKE ?"
            params.append(f"%{q.lower()}%")

        if sort == "shop":
            order_by = f"ORDER BY shop {order}, price_eur ASC"
        elif sort == "name":
            order_by = f"ORDER BY name {order}, price_eur ASC"
        else:
            order_by = f"ORDER BY price_eur {order}, name ASC"

        sql = f"""
            SELECT shop, name, price_eur, price_text, url, inserted_at
            FROM products
            WHERE {where}
            {order_by}
            LIMIT 500
        """
        rows = query_db(sql, params)
        # Last update
        last_row = query_db("SELECT MAX(inserted_at) AS last FROM products", one=True)
        last_update = last_row["last"] if last_row and last_row["last"] else None
        return render_template("results.html", rows=rows, q=q, sort=sort, order=order, last_update=last_update)

    @app.route("/healthz")
    def healthz():
        # Simple health check
        try:
            _ = query_db("SELECT 1")
            return {"status": "ok"}, 200
        except Exception as e:
            return {"status": "error", "msg": str(e)}, 500

    return app

app = create_app()
